#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

#define SDA_PIN D4
#define SCL_PIN D5
#define LED_PIN D1

#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define SCREEN_ADDRESS 0x3C

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

#include <WiFi.h>
#include <PubSubClient.h>

const char* WIFI_SSID = "your wifi name";
const char* WIFI_PASS = "your wifi pass";

const char* AIO_USERNAME = "youraiousername"; 
const char* AIO_KEY      = "your aio key";
const char* MQTT_HOST    = "io.adafruit.com";
const uint16_t MQTT_PORT = 1883;

// Feed/topic
String FEED = "oled-text";
String TOPIC_RAW  = String(AIO_USERNAME) + "/feeds/" + FEED;
String TOPIC_JSON = TOPIC_RAW + "/json";

WiFiClient wifi;
PubSubClient mqtt(wifi);

void blinkLED(int n=2, int on=120, int off=90){
  pinMode(LED_PIN, OUTPUT);
  for(int i=0;i<n;i++){ digitalWrite(LED_PIN,1); delay(on); digitalWrite(LED_PIN,0); delay(off); }
}

void drawTextWrapped(const String& msg){
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  uint8_t col=0,row=0, maxCols=21; // ~6px/char at size=1
  for(size_t i=0;i<msg.length();i++){
    char ch=msg[i]; if(ch=='\r') continue;
    if(ch=='\n'||col>=maxCols){ row++; col=0; if(ch=='\n') continue; }
    if(row>=8) break;
    display.setCursor(col*6, row*8);
    display.write(ch);
    col++;
  }
  display.display();
}

void onMsg(char* topic, byte* payload, unsigned int len) {
  // Copy payload
  String m; m.reserve(len);
  for (unsigned i=0;i<len;i++) m += (char)payload[i];

  // If we got the JSON variant (from REST), pull out the "value" field
  String t(topic);
  if (t.endsWith("/json")) {
    int i = m.indexOf("\"value\"");
    if (i >= 0) {
      int c = m.indexOf(':', i);
      int q1 = m.indexOf('"', c+1);
      int q2 = m.indexOf('"', q1+1);
      if (q1 >= 0 && q2 > q1) m = m.substring(q1+1, q2);
    }
  }

  Serial.printf("MQTT IN [%s] (%u bytes): %s\n", topic, len, m.c_str());
  blinkLED(2);
  drawTextWrapped(m);
}



void connectWiFi() {
  Serial.printf("Connecting Wi-Fi to \"%s\"...\n", WIFI_SSID);
  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  unsigned long t0 = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - t0 < 20000) {
    delay(250);
    Serial.print(".");
  }
  Serial.println();
  if (WiFi.status() == WL_CONNECTED) {
    Serial.print("Wi-Fi OK, IP: ");
    Serial.println(WiFi.localIP());
  } else {
    Serial.println("Wi-Fi FAILED");
  }
}

bool connectMQTT() {
  mqtt.setServer(MQTT_HOST, MQTT_PORT);
  mqtt.setCallback(onMsg);
  String clientId = "xiao-c3-" + String((uint32_t)ESP.getEfuseMac(), HEX);

  Serial.printf("Connecting MQTT to %s:%u as %s\n", MQTT_HOST, MQTT_PORT, clientId.c_str());
  bool ok = mqtt.connect(clientId.c_str(), AIO_USERNAME, AIO_KEY);
  if (!ok) {
    Serial.printf("MQTT connect FAILED, state=%d\n", mqtt.state());
    // Common states: -2=DNS fail, -1=timeout, 1=bad proto, 4=bad user/pass, 5=not auth, 7=unavailable
    return false;
  }
  Serial.println("MQTT connected");
  Serial.println("Subscribing to:");
  Serial.println("  " + TOPIC_RAW);
  Serial.println("  " + TOPIC_JSON);
  mqtt.subscribe(TOPIC_RAW.c_str());
  mqtt.subscribe(TOPIC_JSON.c_str());
  return true;
}

unsigned long lastTry = 0;

void setup() {
  Serial.begin(115200);
  delay(200);
  connectWiFi();
  connectMQTT();
  Serial.println("Now publish to your feed via web (Add Data → Create) or your minimal HTML page.");
  Wire.begin(SDA_PIN, SCL_PIN);
if(!display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)){
  Serial.println("SSD1306 init failed"); while(true){ delay(1000); }
}
display.clearDisplay();
display.setTextSize(1);
display.setTextColor(SSD1306_WHITE);
display.setCursor(0,0);
display.println("MQTT waiting...");
display.display();

}

void loop() {
  if (WiFi.status() != WL_CONNECTED && millis() - lastTry > 3000) {
    lastTry = millis();
    connectWiFi();
  }
  if (!mqtt.connected() && millis() - lastTry > 3000) {
    lastTry = millis();
    connectMQTT();
  }
  mqtt.loop();
}
